/******************************************************************************
* (C) Copyright 2007 by Agilent Technologies, All Rights Reserved.
******************************************************************************/

#ifndef AGTPCIECONTROL_H
#define AGTPCIECONTROL_H

#include "AgtPCIPort.h"
#include "DebugLogger.h"

const AgtValueT PCIE_MAXIMUM_NUMBER_OF_PROBE_BOARDS_COPY = 256; // copy from devpcieintdefs.h

class CAgtPCIEControl
{
  /*
  - Not visible to customer
  - OS independent
  - Instanciated during construction of CAgtPortSelector and CAgtPCIEExerciser.
  There is always only one instance (i.e. CAgtPortSelector and CAgtPCIEExerciser
  get the same instance pointer)
  - Its functions are called from classes CAgtPCIEExerciser and CAgtPortSelector:
  - Calls OS dependent functions, which must be implemented in an OS dependent file. 
  - Maps port handles to OSHandles
  */
public:

  CAgtPCIEControl( void );
  ~CAgtPCIEControl( void );

  // Allow only one instance
  void* operator new( size_t );
  void operator delete( void* );

  void ConnectPort( UInt32 deviceId, AgtPortHandleT* portHandle );
  void OSHandleGet( AgtPortHandleT porthandle, HANDLE * osHandle );
  void DisconnectPort( AgtPortHandleT handle );
  void DeviceIdGet( UInt16 vendId, UInt16 devId, UInt16 index, UInt32* deviceId );

  void SetFunctionNumber( AgtPortHandleT portHandle, UInt8 functionNumber );

  void SetConfSpaceFuncNumAndRegRead( AgtPortHandleT portHandle, UInt32 functionNum, UInt32 address,
    UInt8  size, UInt32& val );
  void SetConfSpaceFuncNumAndRegWrite( AgtPortHandleT portHandle, UInt32 functionNum, UInt32 address,
    UInt8  size, UInt32 val );

  void RegRead( AgtPortHandleT portHandle, UInt32 address, UInt8  size, UInt32& val );
  void RegWrite( AgtPortHandleT portHandle, UInt32 address, UInt8  size, UInt32 val );

  void RegRead( AgtPortHandleT portHandle, UInt32 address, UInt8  size, bool bDoNotLock, bool bDoNotUnlock,
    UInt32& val);
  void RegWrite( AgtPortHandleT portHandle, UInt32 address, UInt8  size, bool bDoNotLock, bool bDoNotUnlock,
    UInt32 val );

  void RegBlockWrite( AgtPortHandleT portHandle, UInt32 address, UInt32 numBytes, UInt32 size, UInt8* psaData );
  void RegBlockRead( AgtPortHandleT portHandle, UInt32 address, UInt32 numBytes, UInt32 size, UInt32* pCount,
    UInt8* psaData );

  void ConfRegDirectRead( AgtPortHandleT portHandle, UInt32 address, UInt32& val );
  void ConfRegDirectWrite( AgtPortHandleT portHandle, UInt32 address, UInt32 val );

  void ISPRegRead( AgtPortHandleT portHandle, UInt32 address, UInt32& val );
  void ISPRegWrite( AgtPortHandleT portHandle, UInt32 address, UInt32 val );

  void ISPRegReadWithFnLock( AgtPortHandleT portHandle, UInt32 address, UInt32& regVal );
  void ISPRegWriteWithFnLock( AgtPortHandleT portHandle, UInt32 address, UInt32 regVal );

  void FPGAWrite( AgtPortHandleT portHandle, UInt32 address, UInt32 val );
  void FPGARead( AgtPortHandleT portHandle, UInt32 address, UInt32 &val );

  bool IsCurMachineLittleEndian( void ) const;
  bool IsISFPGAEnabled( AgtPortHandleT portHandle ) const;
  bool IsGen3Exerciser( AgtPortHandleT portHandle ) const;

  void ConfRegDirectReadWithFnLock( AgtPortHandleT portHandle, UInt32 address, UInt32& val );
  void ConfRegDirectWriteWithFnLock( AgtPortHandleT portHandle, UInt32 address, UInt32 val );

  void GetFunctionLockAndFPGAAccess( AgtPortHandleT portHandle );
  void ReleaseFunctionLockAndFPGAAccess( AgtPortHandleT portHandle, bool bReleaseFPGA );

  void GetFunctionLock( AgtPortHandleT portHandle );
  void ReleaseFunctionLock( AgtPortHandleT portHandle );

  // FPGA Access
  void FPGAAccessRequest( AgtPortHandleT portHandle );
  void FPGAAccessRelease( AgtPortHandleT portHandle );

  // Debugging
  void StartDebugLogging( char* strFileName_p );
  void StopDebugLogging();
  CDebugLogger* GetDebugLogger();

protected:
  // The one and only instance of this class
  static CAgtPCIEControl* S_MyOneControllerInstance;
  // incremented by new, decremented by delete
  static UInt16 S_ReferenceCounter;

  // Maximum number of probe boards supported
  static const UInt16 MAX_DEVS = PCIE_MAXIMUM_NUMBER_OF_PROBE_BOARDS_COPY;
  // Maps a port handle (1-based !!) to an OS handle
  HANDLE OsHandle[MAX_DEVS];

  bool isLittleEndian;

  // Only one instance
  static CDebugLogger* s_pDebugLogger;

protected:

  UInt32 Swap( UInt32 val );

private:
  // Distinguishes between Gen2 and Gen3
  UInt8 mIsGen3Exerciser[MAX_DEVS];
  // For E2960B onwards (for gen3 also) the API port can be disabled 
  UInt8 mISAPIEnabled[MAX_DEVS];
  // For E2960B onwards (for gen3 also) the FPGA port can be disabled
  UInt8 mISFPGAEnabled[MAX_DEVS];

  UInt8 m_uFunctionNum[MAX_DEVS];

  // Last value for size
  //UInt8  mLastSizeControl[MAX_DEVS];
  // Last value for address
  //UInt32 mLastAddressControl[MAX_DEVS];

private:
  void PortAPICheck( HANDLE &osHandle, AgtPortHandleT portHandle );
  void RegisterReadWritePrepare( AgtPortHandleT portHandle, UInt32 address, UInt8 size );

  void RegisterRead( AgtPortHandleT portHandle, UInt32 address, UInt8 size, UInt32 &val );
  void RegisterWrite( AgtPortHandleT portHandle, UInt32 address, UInt8 size, UInt32 val );

  //void HandshakeGen2Gen3( HANDLE myOSHandle );
  void SanityCheck( HANDLE myOSHandle, UInt32 val );
  void SanityCheck( AgtPortHandleT portHandle, UInt32 val );

  void FPGAAckClear( AgtPortHandleT portHandle );
  //void ISPValueRegRead( AgtPortHandleT portHandle, UInt32& regVal );
  //void ISPValueRegWrite( AgtPortHandleT portHandle, UInt32 regVal );

  void ISPRegWriteVerify( AgtPortHandleT portHandle, UInt32 address, UInt32 val );

  //void FPGAWaitAck( AgtPortHandleT portHandle, bool waitClear );

  void SetConfSpaceFuncNum( AgtPortHandleT portHandle, UInt32 functionNum );

  // Helper
  void WaitMilliSeconds( int msTime );
  UInt32 swapDW( const UInt32& val );
};

#endif